extends Node

signal tile_changed
signal player_found_enemy
signal player_opened_door
signal player_gained_xp
signal player_found_stairs
signal cleared

var room_width = 8
var room_height = 8
var room_level = 1
var cleared = false
var tiles = []
var number_of_monsters = 1
var monster_base_damage = 4
var monster_min_damage = 0
var monster_max_damage = 16
var player = null

var monster_textures = [
	"monster_ghost.svg",
	"monster_skeleton.svg",
	"monster_devil.svg",
	"monster_knight.svg"
]

var monster_names = {
	"monster_ghost.svg": "Ghost",
	"monster_skeleton.svg": "Skeleton",
	"monster_devil.svg": "Devil",
	"monster_knight.svg": "Knight"
}

class Tile:
	var explored = false
	var monster = null
	var door = null
	var next_floor = false

class Monster:
	var position = Vector2()
	var texture = null
	var name = "Unknown"
	var level = 1
	var alive = true

class Door:
	var hidden = true
	var player_target = Vector2()
	var floor_target = Vector2()

func _ready():
	pass

func set_player(p):
	player = p

func create_new():
	player.position.x = 7
	player.position.y = 4
	create_room()
	redraw_room()

func random_position():
	var pos = Vector2()
	pos.x = randi() % room_width
	pos.y = randi() % room_height
	return pos

func init_room():
	for x in range(room_width):
		if len(tiles) <= x:
			tiles.append([])
		for y in range(room_height):
			var tile = Tile.new()
			tiles[x].append(tile)

func create_room():
	var monsters = create_monsters()	
	for x in range(room_width):
		for y in range(room_height):
			for monster in monsters:
				if monster.position.x == x and monster.position.y == y:
					tiles[x][y].monster = monster

func create_monsters():
	var monsters = []
	var monsters_left = number_of_monsters
	while monsters_left > 0:
		var pos = random_position()
		if tiles[pos.x][pos.y].door != null:
			continue
		var monster_exists = false
		for monster in monsters:
			if pos.x == monster.position.x and pos.y == monster.position.y:
				monster_exists = true
				break
		if monster_exists:
			continue
		var monster = random_monster()
		monster.position = pos
		monsters.append(monster)
		monsters_left = monsters_left - 1
	return monsters

func random_monster():
	var i = randi() % len(monster_textures)
	var texture_name = monster_textures[i]
	var monster = Monster.new()
	monster.name = monster_names[texture_name]
	monster.texture = load("res://%s" % texture_name)
	var min_level = room_level - 1
	if min_level < 1:
		min_level = 1
	var max_level = room_level + 1
	var monster_level = randi() % (1 + max_level - min_level)
	monster_level += min_level
	monster.level = monster_level
	return monster

func redraw_room():
	var tilemap = get_node("TileMap")
	var tileset = tilemap.tile_set
	for x in range(room_width):
		for y in range(room_height):
			var tile = tileset.find_tile_by_name("floor_unknown")
			if player.position.x == x and player.position.y == y:
				tile = tileset.find_tile_by_name("player")
			elif tiles[x][y].door != null and tiles[x][y].door.hidden == false:
				tile = tileset.find_tile_by_name("door")
			elif cleared and tiles[x][y].next_floor:
				tile = tileset.find_tile_by_name("black")
			elif tiles[x][y].explored:
				if tiles[x][y].monster != null:
					tile = tileset.find_tile_by_name("mob_debug")
				else:
					tile = tileset.find_tile_by_name("floor_safe")
					var num_monsters = tile_close_monsters_num(x, y)
					if num_monsters > 0:
						var tile_name = "floor_%d" % num_monsters
						tile = tileset.find_tile_by_name(tile_name)
			#elif tiles[x][y].monster != null:
			#	tile = tileset.find_tile_by_name("mob_debug")
			tilemap.set_cell(x, y, tile)

func player_go_left():
	if player.fighting:
		return
	player.position.x -= 1
	if player.position.x < 0:
		player.position.x = 0
		return
	player_moved()

func player_go_right():
	if player.fighting:
		return
	player.position.x += 1
	if player.position.x >= room_width:
		player.position.x = room_height - 1
		return
	player_moved()

func player_go_up():
	if player.fighting:
		return
	player.position.y -= 1
	if player.position.y < 0:
		player.position.y = 0
		return
	player_moved()

func player_go_down():
	if player.fighting:
		return
	player.position.y += 1
	if player.position.y >= room_height:
		player.position.y = room_height -1
		return
	player_moved()

func player_moved():
	var x = player.position.x
	var y = player.position.y
	var stop = explore_tile(x, y)
	if not stop:
		emit_signal_player_moved(x, y)
		if not cleared:
			if all_tiles_explored():
				cleared = true
				player_level_up(room_level * (player.next_level * 0.25))
				emit_signal("cleared")
				remove_monsters()
				show_doors()
		redraw_room()

func player_fight_monster():
	if not player.fighting:
		return
	
	var x = player.position.x
	var y = player.position.y
	var tile = tiles[x][y]
	if not tile.monster:
		return
	
	var monster_player_level_diff = tile.monster.level - player.level
	if monster_player_level_diff < 0:
		monster_player_level_diff = 0
	var min_damage = monster_base_damage * tile.monster.level
	min_damage += monster_base_damage * monster_player_level_diff
	min_damage += monster_min_damage
	
	var max_damage = min_damage + monster_max_damage
	var damage = randi() % (1 + max_damage - min_damage)
	damage += min_damage
	
	player.health -= damage
	if player.health > 0:
		player_level_up(tile.monster.level * 2)
		tile.monster.alive = false
		tile.monster = null
		redraw_room()
		emit_signal_player_moved(x, y)

func player_level_up(xp):
	player.current_xp += xp
	var gained_level = false
	if player.current_xp >= player.next_level:
		player.level += 1
		var diff = player.current_xp - player.next_level
		player.next_level += player.next_level * 0.25
		player.current_xp = diff
		gained_level = true
	emit_signal("player_gained_xp", gained_level)

func all_tiles_explored():
	for x in range(room_width):
		for y in range(room_height):
			if tiles[x][y].monster == null and tiles[x][y].explored == false:
				return false
	return true

func remove_monsters():
	for x in range(room_width):
		for y in range(room_height):
			if tiles[x][y].monster != null:
				tiles[x][y].monster = null
				tiles[x][y].explored = true

func show_doors():
	for x in range(room_width):
		for y in range(room_height):
			if tiles[x][y].door != null:
				tiles[x][y].door.hidden = false

func emit_signal_player_moved(x, y):
	var tilemap = get_node("TileMap")
	var tileset = tilemap.tile_set
	if tiles[x][y].monster != null:
		var monster = tiles[x][y].monster
		emit_signal("tile_changed", monster.name, monster.texture)
	else:
		var tile = tileset.find_tile_by_name("floor_safe")
		var tile_name = "Empty"
		var num_monsters = tile_close_monsters_num(x, y)
		if num_monsters > 0:
			tile_name = "Danger"
			tile = tileset.find_tile_by_name("floor_%d" % num_monsters)
		var tile_texture = tileset.tile_get_texture(tile)
		emit_signal("tile_changed", tile_name, tile_texture)

func explore_tile(x, y, open_door=true):
	var tile = tiles[x][y]
	if tile.monster != null:
		emit_signal("player_found_enemy", tile.monster)
	elif tile.door != null and tile.door.hidden == false and open_door:
		emit_signal("player_opened_door", tile.door)
		return true
	elif cleared and tile.next_floor:
		emit_signal("player_found_stairs")
	tiles[x][y].explored = true
	return false

func tile_close_monsters_num(x, y):
	var monsters = 0
	var start = Vector2()
	var end = Vector2()
	
	start.x = x - 1
	start.y = y - 1
	
	if start.x < 0:
		start.x = 0
	if start.y < 0:
		start.y = 0
	
	end.x = x + 1
	end.y = y + 1
	
	if end.x >= room_width:
		end.x = room_width - 1
	if end.y >= room_height:
		end.y = room_height - 1
	
	x = start.x
	while x <= end.x:
		y = start.y
		while y <= end.y:
			if tiles[x][y].monster != null:
				monsters += 1
			y += 1
		x += 1
	
	return monsters

func add_door(pos, ptarget, ftarget):
	var door = Door.new()
	door.player_target = ptarget
	door.floor_target = ftarget
	tiles[pos.x][pos.y].door = door

func add_next_floor():
	var pos = null
	while true:
		pos = random_position()
		if tiles[pos.x][pos.y].door != null:
			continue
		elif tiles[pos.x][pos.y].monster != null:
			continue
		break
	tiles[pos.x][pos.y].next_floor = true
